/*
Copyright (C) 2007-2024 Victor Matei Petrescu

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

typedef float T3REALD[3]; /*looks strange, but 'T3REALD x' is now 'float x[3]'*/


typedef struct _objgeom{
 int ntr; /*number of triangles (collision)*/
 T3REALD *p1; /*p1[0]=x, p1[1]=y, p1[2]=z*/
 T3REALD *p2;
 T3REALD *p3; /*vertices of triangles (collision) in local system of object type*/
} objgeom; /*object type*/


typedef struct _gameobj{
 int otyp; /*number of object type in the global array (fceglob)*/
 int ntr; /*number triangles (collision)*/
 float vx[4];
 float vy[4];
 float vz[4]; /*directions of local axes in global coordinates - v[1]-v[0]=i,
               v[2]-v[0]=j, v[3]-v[0]=k; v[0] - position (origin of local sist.);
               in local coord. vx[1]=1, vy[1]=0, vz[1]=0; vx[2]=0, vy[2]=1, vz[2]=0;
               rotation and translation of object only affect vx, vy, vz;
               triangles are calculated from vx, vy and vz at rendering*/
 int lev; /*level*/
 float mu; /*friction*/
} gameobj; /*3D object*/


typedef struct _vhc{
 int nob; /*number of objects in vehicle*/
 int nj; /*number of joints*/

 int oid[21]; /*number of object (rendering and collision) in 'objs' array from 'simcar.c'*/
 int bid[21]; /*particles*/
 joint *jid[21]; /*joints*/
 float jax[21][3]; /*joint rotation points*/

 int jfc[21]; /*function of object attached to joint i (see 'ofc' below)*/
 int ofc[21]; /*function of object i: 1-car; 2-trailer; 3-motor wheel; 4-steering wheel;
   5-motor and steering wheel; 6-passive wheel; 7-trailer wheel*/
 float accel;
 float brake; /*acceleration and brake torques/wheel*/
 float spring[2]; /*spring[0] vertical suspension stiffness, spring[1] horizontal stiffness*/
 float damper[2]; /*vertical and horizontal suspension damping coef.*/

 float drag;
 float downforce; /*these are multiplied with (speed^2)*/
 float rocket[2]; /*rocket thrust and time available until fuel exhausted*/

 float power; /*kW*/

 float cam[3]; /*camera position*/
} vhc; /*vehicle*/


/*----GLOBAL VARIABLE----*/
objgeom *refglob; /*object types*/


#if REPLAY==1
typedef struct _repdata{
 int car;
 int cno; /*number of objects in vehicle*/
 int track;
 float ct; /*current time or time since last frame saved*/
 float tf; /*desired interval between frames*/
 int nf; /*number of frames saved*/
 int na; /*number of frames for which memory was allocated (nf<=na)*/
 float *t; /*t[i]-time at frame i; 0<=i<nf*/
 float *p; /*p[(cno*i+j)*9+k]-data for object j at frame i;
   k=0,1,2-position; k=3..8-first 2 lines of rotation matrix*/
} repdata; /*replay data*/

repdata rpglob;


void freereplay(){
 free(rpglob.t); free(rpglob.p);
 rpglob.t=NULL; rpglob.p=NULL;
 rpglob.na=0;
}


void initreplay(int car,int track,int cno){
 if(rpglob.t!=NULL){freereplay();} /*rpglob.t and rpglob.p initialised in main()*/
 if(!(rpglob.t=(float *)malloc((REPTIME*REPFPS)*sizeof(float)))){printf("Out of memory\r\n"); exit(1);}
 if(!(rpglob.p=(float *)malloc((9*cno*REPTIME*REPFPS)*sizeof(float)))){printf("Out of memory\r\n"); exit(1);}
 rpglob.car=car; rpglob.track=track;
 rpglob.cno=cno; /*number of objects in vehicle*/
 rpglob.nf=0; rpglob.na=REPTIME*REPFPS;
 rpglob.ct=0.0; rpglob.tf=1.0/(float)REPFPS;
}


/*write replay data from rpglob to file repf*/
int writerepf(char *repf){
  char text[64];
  int i,j,k;
  FILE *f;
sprintf(text,"replays/%s",repf);
if(!(f=fopen(text,"w"))){printf("Could not open '%s'\r\n",repf); return 0;}

fprintf(f,"%d %d %d %d\r\n",rpglob.car,rpglob.cno,rpglob.track,rpglob.nf);

for(i=0;i<rpglob.nf;i++){
  fprintf(f,"%1.3f ",rpglob.t[i]);
  for(j=0;j<rpglob.cno;j++){
    for(k=0;k<9;k++){
      fprintf(f,"%1.3f ",rpglob.p[(rpglob.cno*i+j)*9+k]);
    }
  } fprintf(f,"\r\n");
}

fclose(f); return 1;
}


/*read replay data from file repf*/
int readrepf(char *repf){
  char text[64];
  int i,j,k;
  FILE *f;
sprintf(text,"replays/%s",repf);
if(!(f=fopen(text,"r"))){printf("Could not open '%s'\r\n",repf); return 0;}
fscanf(f,"%d %d %d %d",&(rpglob.car),&(rpglob.cno),&(rpglob.track),&(rpglob.nf));

initreplay(rpglob.car,rpglob.track,rpglob.cno);

rewind(f);
fscanf(f,"%d %d %d %d",&(rpglob.car),&(rpglob.cno),&(rpglob.track),&(rpglob.nf));

for(i=0;i<rpglob.nf;i++){
  fscanf(f,"%f ",&(rpglob.t[i]));
  for(j=0;j<rpglob.cno;j++){
    for(k=0;k<9;k++){
      fscanf(f,"%f ",&(rpglob.p[(rpglob.cno*i+j)*9+k]));
    }
  }
}

fclose(f); return 1;
}
#endif


/*put vector k on 3rd line of transposed rotation matrix, which contains i and j on the first
2 lines, OR put iz, jz and kz in matrix with ix, jx, kx, iy, jy and ky on the first 2 lines*/
void fillRotM(float *rt){
 rt[6]=rt[1]*rt[5]-rt[2]*rt[4];
 rt[7]=rt[2]*rt[3]-rt[0]*rt[5];
 rt[8]=rt[0]*rt[4]-rt[1]*rt[3];
}


/*find rotation axis and angle between matrices r1 and r2*/
float findRotM(float *r1,float *r2,float *ax){
  float ang;
ax[0]=r1[3]*r2[6]-r1[6]*r2[3]+r1[4]*r2[7]-r1[7]*r2[4]+r1[5]*r2[8]-r1[8]*r2[5];
ax[1]=r1[6]*r2[0]-r1[0]*r2[6]+r1[7]*r2[1]-r1[1]*r2[7]+r1[8]*r2[2]-r1[2]*r2[8];
ax[2]=r1[0]*r2[3]-r1[3]*r2[0]+r1[1]*r2[4]-r1[4]*r2[1]+r1[2]*r2[5]-r1[5]*r2[2];
ang=asin(0.5*sqrt(ax[0]*ax[0]+ax[1]*ax[1]+ax[2]*ax[2]));
return ang;
}


/*function which translates an object by x,y and z*/
void translat(gameobj *objs,float x,float y,float z){
  int i;
for(i=0;i<=3;i++){
  objs->vx[i]+=x;
  objs->vy[i]+=y;
  objs->vz[i]+=z;
}
}


/*function which rotates object around an axis parallel with z which intersects
plan xOy in point with coordinates x and y*/
void rotatz(gameobj *objs,float x,float y,float tt){
  int i;
  float xtm,sintt,costt;

sintt=sin(tt);costt=cos(tt);

for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*costt-(objs->vy[i]-y)*sintt;
  objs->vy[i]=y+(objs->vy[i]-y)*costt+(xtm-x)*sintt;
}
}


/*function which rotates object around an axis parallel with y which intersects
plan xOz in point with coordinates x and z*/
void rotaty(gameobj *objs,float x,float z,float tt){
  int i;
  float xtm,sintt,costt;

sintt=sin(tt);costt=cos(tt);

for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*costt+(objs->vz[i]-z)*sintt;
  objs->vz[i]=z+(objs->vz[i]-z)*costt-(xtm-x)*sintt;
}
}


/*function which rotates object around an axis parallel with x which intersects
plan xOy in point with coordinates x and y*/
void rotatx(gameobj *objs,float y,float z,float tt){
  int i;
  float ytm,sintt,costt;

sintt=sin(tt);costt=cos(tt);

for(i=0;i<=3;i++){
  ytm=objs->vy[i];
  objs->vy[i]=y+(objs->vy[i]-y)*costt-(objs->vz[i]-z)*sintt;
  objs->vz[i]=z+(objs->vz[i]-z)*costt+(ytm-y)*sintt;
}
}


/*function which rotates an object around an axis determined by A(x,y,z) and B(xb,yb,zb)*/
void rotab(gameobj *objs,float x,float y,float z,float xb,float yb,float zb,float tt){
  int i;
  float xtm,
sinalf=0,cosalf=0,sinbt=0,cosbt=0,sintt,costt,
len2,len1, /*length of segment AB and its projection on plan xOy*/
abx,aby,abz, /*projections of segment AB on the 3 axes*/
thres=1e-5; /*min. accepted value for len1*/

abx=xb-x; aby=yb-y; abz=zb-z;
len1=sqrt(abx*abx+aby*aby);
len2=sqrt(abx*abx+aby*aby+abz*abz);

if(len1>thres){
sinalf=aby/len1; cosalf=abx/len1; sinbt=len1/len2; cosbt=abz/len2;
}else{
if(abz<0){tt=-tt;}
}

sintt=sin(tt);costt=cos(tt);

	if(len1>thres){
/*1 - rotation with (-alpha) around axis z*/
for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*cosalf+(objs->vy[i]-y)*sinalf;
  objs->vy[i]=y+(objs->vy[i]-y)*cosalf-(xtm-x)*sinalf;
}

/*2 - rotation with (-beta) around axis y*/
for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*cosbt-(objs->vz[i]-z)*sinbt;
  objs->vz[i]=z+(objs->vz[i]-z)*cosbt+(xtm-x)*sinbt;
}
	}
/*3 - rotation with theta around axis z*/
for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*costt-(objs->vy[i]-y)*sintt;
  objs->vy[i]=y+(objs->vy[i]-y)*costt+(xtm-x)*sintt;
}

	if(len1>thres){
/*4 - rotation with beta around axis y*/
for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*cosbt+(objs->vz[i]-z)*sinbt;
  objs->vz[i]=z+(objs->vz[i]-z)*cosbt-(xtm-x)*sinbt;
}

/*5 - rotation with alpha around axis z*/
for(i=0;i<=3;i++){
  xtm=objs->vx[i];
  objs->vx[i]=x+(objs->vx[i]-x)*cosalf-(objs->vy[i]-y)*sinalf;
  objs->vy[i]=y+(objs->vy[i]-y)*cosalf+(xtm-x)*sinalf;
}
	}
}
