/*
Copyright (C) 2007-2018 Victor Matei Petrescu

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "../grconfig.h"
#include "../graphics.h"

typedef struct _triangl3d{
  float x1; float y1;float z1;
  float x2; float y2;float z2;
  float x3; float y3;float z3;
  float red; float green; float blue;
  Uint8 cull; /*first bit(&1):1-cull;0-no cull;second bit(&2):1-fullbright*/
} triangl3d;

typedef struct _mod3d{
  int nt[2]; /*nt[0]-nr.of triangles;nt[1]-nr.of triangles for which memory was allocated*/
  triangl3d *tri;
  float radius;
} mod3d;

typedef struct _obj3d{
  int nmod; /*model nr.*/
  float pos[3];
  float rot[9]; /*rotation matrix*/
  float radius;
} obj3d;

typedef struct _lightpr{
  float amb;
  float hd;
  float dir; /*intensities*/
  float dx;
  float dy;
  float dz; /*direction*/
} lightpr; /*light parameters*/


extern unsigned int RGLOB_width,RGLOB_height;
extern void **RGLOB_lpinc,**RGLOB_lpdec; /*[i]-pointer to the start of line i*/
extern char RGLOB_initgr; /*initgr=1-graphics initialized;*/
extern Uint32 RGLOB_color[2];

extern SDL_Surface *RGLOB_screen;
extern const SDL_PixelFormatDetails *RGLOB_pixformdet;

char RGLOB_init3d=0;
float RGLOB_zmin,RGLOB_zmax,RGLOB_zfd,*RGLOB_zbuf, /*Zbuffer*/
      RGLOB_tgh,RGLOB_tgv;
lightpr RGLOB_light[2];
mod3d *RGLOB_mod3d;
obj3d *RGLOB_obj3d;
unsigned int RGLOB_nrmd3d[2]={0,0},RGLOB_nrobj3d[2]={0,0}; /*nr.of models and objects*/
obj3d RGLOB_cam; /*camera*/


int chkname(const char *s){
  int i=0;
  char nul[4]={'n','u','l','\0'};
  while((i<4)&&(s[i]==nul[i])){i++;}
  if(i==4){return 0;}else{return 1;}
}


void setlight3d(float amb,float hd,float dir,float dx,float dy,float dz){
  int i;

for(i=0;i<2;i++){
  RGLOB_light[i].amb=amb; RGLOB_light[i].hd=hd; RGLOB_light[i].dir=dir;
  RGLOB_light[i].dx=dx; RGLOB_light[i].dy=dy; RGLOB_light[i].dz=dz;
} /*light[1] will be rotated*/
}


void findplan(triangl3d *tri,float *a,float *b,float *c,float *d){
  float a1,b1,c1,a2,b2,c2;
a1=tri->x1 - tri->x2; b1=tri->y1 - tri->y2; c1=tri->z1 - tri->z2;
a2=tri->x1 - tri->x3; b2=tri->y1 - tri->y3; c2=tri->z1 - tri->z3;
*a = b1*c2-b2*c1; *b = a2*c1-a1*c2; *c = a1*b2-a2*b1;
*d = (*a)*tri->x1 + (*b)*tri->y1 + (*c)*tri->z1; /*ax+by+cz=d*/
}


int initgraph3d(float fov,float zmin,float zmax){
  unsigned int i,n;
  float izmax;

if(RGLOB_init3d==1){printf("initgraph3d(): 3D already initialized\r\n"); return 0;}
if(RGLOB_initgr==0){printf("initgraph3d(): Screen not initialized\r\n"); exit(1);}
if(zmin>zmax){printf("initgraph3d(): zmin (%1.3f) > zmax (%1.3f)\r\n",zmin,zmax); exit(1);}

#if DOUBLEPIX==0
  n=RGLOB_width*RGLOB_height;
#elif DOUBLEPIX==1
  n=(RGLOB_width>>1)*(RGLOB_height>>1);
#else
  printf("initgraph3d(): DOUBLEPIX should be set to 0 or 1 in 'grconfig.h'\r\n"); exit(1);
#endif

if(!(RGLOB_zbuf=(float *)malloc(n*sizeof(float)))){printf("initgraph3d(): Out of memory\r\n"); exit(1);}

izmax=1.0f/zmax;
for(i=0;i<n;i++){RGLOB_zbuf[i]=izmax;}
RGLOB_zmin=zmin; RGLOB_zmax=zmax;
RGLOB_zfd=0.5f*(float)RGLOB_width/tan(fov*0.00872665f);
RGLOB_tgh=0.5f*(float)RGLOB_width/((float)RGLOB_zfd);
RGLOB_tgv=0.5f*(float)RGLOB_height/((float)RGLOB_zfd);

#if ASPCOR==1
  RGLOB_zfd=(int)((float)RGLOB_zfd/WIDTHFACTOR);
  RGLOB_tgv*=WIDTHFACTOR;
#elif ASPCOR!=0
  printf("Error: ASPCOR should be set to 0 or 1 in 'grconfig.h'\r\n"); exit(1);
#endif

for(i=0;i<9;i++){RGLOB_cam.rot[i]=0.0f;}
for(i=0;i<3;i++){RGLOB_cam.pos[i]=0.0f; RGLOB_cam.rot[i*3+i]=1.0f;}
setlight3d(0.3,0.4,1.0,0.0,-1.0,0.0); /*default light*/

RGLOB_init3d=1;
return 1;
}


int dtriangle3d(triangl3d *tri){
  int i,j,k,l,kmin,invs,imin,imax,jmin,jmax;
  unsigned int width,height;
  float tmp,tmp2,slope2,slope3,slope4,xcr,ycr,ystart,yend,thres=1.0e-5f,
        x[3],y[3],z[3],a,b,c,d,zf,izf,
        aizf,bizf,id,idbizf,
        a1,dist;
  triangl3d dtri[2];
  lightpr *light;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpinc;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpinc;
  #endif

if(RGLOB_init3d==0){printf("dtriangle3d(): 3D not initialized\r\n"); exit(1);}

if((tri->z1 <= RGLOB_zmin)&&(tri->z2 <= RGLOB_zmin)&&(tri->z3 <= RGLOB_zmin)){return 0;}
if((tri->z1 >= RGLOB_zmax)&&(tri->z2 >= RGLOB_zmax)&&(tri->z3 >= RGLOB_zmax)){return 0;}
  if((tri->x1 > RGLOB_tgv*tri->z1)&&(tri->x2 > RGLOB_tgv*tri->z2)&&(tri->x3 > RGLOB_tgv*tri->z3)){return 0;}
  if((tri->y1 > RGLOB_tgh*tri->z1)&&(tri->y2 > RGLOB_tgh*tri->z2)&&(tri->y3 > RGLOB_tgh*tri->z3)){return 0;}
if((tri->x1 < -RGLOB_tgv*tri->z1)&&(tri->x2 < -RGLOB_tgv*tri->z2)&&(tri->x3 < -RGLOB_tgv*tri->z3)){return 0;}
if((tri->y1 < -RGLOB_tgh*tri->z1)&&(tri->y2 < -RGLOB_tgh*tri->z2)&&(tri->y3 < -RGLOB_tgh*tri->z3)){return 0;}

x[0]=tri->x1 - tri->x2; y[0]=tri->y1 - tri->y2; z[0]=tri->z1 - tri->z2;
x[1]=tri->x1 - tri->x3; y[1]=tri->y1 - tri->y3; z[1]=tri->z1 - tri->z3;
a=y[0]*z[1]-y[1]*z[0]; b=x[1]*z[0]-x[0]*z[1]; c=x[0]*y[1]-x[1]*y[0];
d=a*tri->x1 + b*tri->y1 + c*tri->z1;

#if DOUBLEPIX==1
  width=RGLOB_width>>1; height=RGLOB_height>>1; zf=-0.5f*RGLOB_zfd;
#else
  width=RGLOB_width; height=RGLOB_height; zf=-RGLOB_zfd;
#endif

/*start lighting and backface culling*/
tmp=a*tri->x1 + b*tri->y1 + c*tri->z1; /*dot product*/
if(((tri->cull)&1)&&(tmp<0.0f)){return 0;} /*backface culling*/

izf=1.0f/zf; aizf=a*izf; bizf=b*izf;
if((d>thres)||(d<(-thres))){id=1.0f/d;}else{id=1.0e5f;}

#if ASPCOR==1
  bizf/=WIDTHFACTOR;
#endif

idbizf=id*bizf;

if((tri->cull)&2){ /*fullbright*/
  a1=1.0f;
}else{
  light=&(RGLOB_light[1]);
  dist=sqrt(a*a+b*b+c*c);
  a1=(light->dx * a + light->dy * b + light->dz * c)/dist;
  if(tmp>=0.0f){
    if(a1<0.0f){a1=0.0f;}
  }else{
    if(a1<0.0f){a1=-a1;}else{a1=0.0f;}
  }
  a1=light->amb + fabs(light->hd * c/dist) + light->dir * a1;
}

if(a1<thres){a1=thres;}

i=(int)(a1 * tri->red); j=(int)(a1 * tri->green); k=(int)(a1 * tri->blue);
if(i>255){i=255;}
if(j>255){j=255;}
if(k>255){k=255;}

setcolor((unsigned char)i,(unsigned char)j,(unsigned char)k);
/*finished lighting and backface culling*/

j=0;

if((tri->z1 > RGLOB_zmin)&&(tri->z2 > RGLOB_zmin)&&(tri->z3 > RGLOB_zmin)){
  dtri[j]=(*tri); j++;
}else{
  x[0]=tri->x1; x[1]=tri->x2; x[2]=tri->x3;
  y[0]=tri->y1; y[1]=tri->y2; y[2]=tri->y3;
  z[0]=tri->z1; z[1]=tri->z2; z[2]=tri->z3;

  invs=1;

  for(k=0;k<2;k++){kmin=k;
    for(l=k+1;l<3;l++){
      if(z[l]<z[kmin]){kmin=l;}
    }
    if(kmin!=k){
      invs*=-1;
      tmp=x[k]; x[k]=x[kmin]; x[kmin]=tmp;
      tmp=y[k]; y[k]=y[kmin]; y[kmin]=tmp;
      tmp=z[k]; z[k]=z[kmin]; z[kmin]=tmp;
    }
  } /*ordonat varfuri dupa z*/

  if((z[0]<=RGLOB_zmin)&&(z[1]<=RGLOB_zmin)){
    tmp=(RGLOB_zmin-z[0])/(z[2]-z[0]);
    dtri[j].x1=tmp*(x[2]-x[0])+x[0];
    dtri[j].y1=tmp*(y[2]-y[0])+y[0];
    tmp=(RGLOB_zmin-z[1])/(z[2]-z[1]);
    dtri[j].x2=tmp*(x[2]-x[1])+x[1];
    dtri[j].y2=tmp*(y[2]-y[1])+y[1];
    dtri[j].x3=x[2];
    dtri[j].y3=y[2];
    dtri[j].z1=dtri[j].z2=RGLOB_zmin;
    dtri[j].z3=z[2];
    dtri[j].red=tri->red; dtri[j].green=tri->green; dtri[j].blue=tri->blue;
    dtri[j].cull=tri->cull;
    if(invs==(-1)){
      tmp2=dtri[j].x1; dtri[j].x1=dtri[j].x2; dtri[j].x2=tmp2;
      tmp2=dtri[j].y1; dtri[j].y1=dtri[j].y2; dtri[j].y2=tmp2;
      tmp2=dtri[j].z1; dtri[j].z1=dtri[j].z2; dtri[j].z2=tmp2;
    }
    j++;
  }else{
    tmp=(RGLOB_zmin-z[0])/(z[1]-z[0]);
    dtri[j].x1=tmp*(x[1]-x[0])+x[0];
    dtri[j].y1=tmp*(y[1]-y[0])+y[0];
    dtri[j].x2=x[1];
    dtri[j].y2=y[1];
    dtri[j].x3=x[2];
    dtri[j].y3=y[2];
    dtri[j].z1=RGLOB_zmin;
    dtri[j].z2=z[1];
    dtri[j].z3=z[2];
    dtri[j].red=tri->red; dtri[j].green=tri->green; dtri[j].blue=tri->blue;
    dtri[j].cull=tri->cull;
    if(invs==(-1)){
      tmp2=dtri[j].x1; dtri[j].x1=dtri[j].x2; dtri[j].x2=tmp2;
      tmp2=dtri[j].y1; dtri[j].y1=dtri[j].y2; dtri[j].y2=tmp2;
      tmp2=dtri[j].z1; dtri[j].z1=dtri[j].z2; dtri[j].z2=tmp2;
    }
    j++;

    dtri[j].x1=tmp*(x[1]-x[0])+x[0];
    dtri[j].y1=tmp*(y[1]-y[0])+y[0];
    tmp=(RGLOB_zmin-z[0])/(z[2]-z[0]);
    dtri[j].x2=tmp*(x[2]-x[0])+x[0];
    dtri[j].y2=tmp*(y[2]-y[0])+y[0];
    dtri[j].x3=x[2];
    dtri[j].y3=y[2];
    dtri[j].z1=dtri[j].z2=RGLOB_zmin;
    dtri[j].z3=z[2];
    dtri[j].red=tri->red; dtri[j].green=tri->green; dtri[j].blue=tri->blue;
    dtri[j].cull=tri->cull;
    if(invs==(1)){
      tmp2=dtri[j].x1; dtri[j].x1=dtri[j].x2; dtri[j].x2=tmp2;
      tmp2=dtri[j].y1; dtri[j].y1=dtri[j].y2; dtri[j].y2=tmp2;
      tmp2=dtri[j].z1; dtri[j].z1=dtri[j].z2; dtri[j].z2=tmp2;
    }
    j++;
  }
}

for(k=0;k<j;k++){
  x[0]=dtri[k].x1*zf/dtri[k].z1;
  x[1]=dtri[k].x2*zf/dtri[k].z2;
  x[2]=dtri[k].x3*zf/dtri[k].z3;
  y[0]=dtri[k].y1*zf/dtri[k].z1;
  y[1]=dtri[k].y2*zf/dtri[k].z2;
  y[2]=dtri[k].y3*zf/dtri[k].z3;

  if(x[1]>x[2]){tmp=y[1]; y[1]=y[2]; y[2]=tmp; tmp=x[1]; x[1]=x[2]; x[2]=tmp;}
  if(x[0]>x[1]){
    tmp=y[0]; y[0]=y[1]; y[1]=tmp; tmp=x[0]; x[0]=x[1]; x[1]=tmp;
    if(x[1]>x[2]){tmp=y[1]; y[1]=y[2]; y[2]=tmp; tmp=x[1]; x[1]=x[2]; x[2]=tmp;}
  } /*ordered triangle vertices*/

  imax=(int)((int)(height>>1)+x[2]); imin=(int)((int)(height>>1)+x[0]+0.99999f);
  if(imin<0){imin=0;}
  if(imax>((int)height-1)){imax=(int)height-1;} /*found triangle limits*/

  if((x[1]-x[0])>thres){slope2=(y[1]-y[0])/(x[1]-x[0]);}else{slope2=(y[1]-y[0])/thres;}
  if((x[2]-x[0])>thres){slope3=(y[2]-y[0])/(x[2]-x[0]);}else{slope3=(y[2]-y[0])/thres;}
  if((x[2]-x[1])>thres){slope4=(y[2]-y[1])/(x[2]-x[1]);}else{slope4=(y[2]-y[1])/thres;}

  for(i=imin;i<=imax;i++){
    xcr=-(int)(height>>1)+i;
    if(xcr<x[1]){
      ystart=(slope2*(xcr-x[0])+y[0]);
      yend=(slope3*(xcr-x[0])+y[0]);}
    else{
      ystart=(slope4*(xcr-x[1])+y[1]);
      yend=(slope3*(xcr-x[0])+y[0]);}
    if(yend<ystart){tmp=ystart;ystart=yend;yend=tmp;}

    #if ASPCOR==1
      ystart*=WIDTHFACTOR;
      yend*=WIDTHFACTOR;
    #endif

    jmin=(int)(-yend+(int)(width>>1)+0.99f); jmax=(int)(-ystart+(int)(width>>1));
    if(jmin<0){jmin=0;}
    if(jmax>((int)width-1)){jmax=(int)width-1;}

    l=i*width+jmin;
    ycr=(int)(width>>1)-jmin+1;
    dist=id*(aizf*xcr+bizf*ycr+c);

  #if DOUBLEPIX==0
    ptrmax=pl[i]+jmax;
    for(ptr=pl[i]+jmin;ptr<=ptrmax;ptr++)
  #else
    ptrmax=pl[i+i]+jmax+jmax;
    for(ptr=pl[i+i]+jmin+jmin;ptr<=ptrmax;ptr+=2)
  #endif
    {
      dist-=idbizf;
      if(RGLOB_zbuf[l]<dist){
        RGLOB_zbuf[l]=dist;
        *ptr=RGLOB_color[0];
      }
      l++;
    }
  }
}

return j;
}


/*finds size of the model*/
float eval_mod(unsigned int i){
  unsigned int n;
  float xmin,xmax,ymin,ymax,zmin,zmax;
  mod3d *m3d;

if(i>=RGLOB_nrmd3d[0]){printf("eval_mod(): model %d not existr\r\n",i); exit(1);}

m3d=&(RGLOB_mod3d[i]);
n=m3d->nt[0];

xmin=xmax=m3d->tri[0].x1;
ymin=ymax=m3d->tri[0].y1;
zmin=zmax=m3d->tri[0].z1;

for(i=0;i<n;i++){
  if(xmin > m3d->tri[i].x1){xmin=m3d->tri[i].x1;}
  if(xmin > m3d->tri[i].x2){xmin=m3d->tri[i].x2;}
  if(xmin > m3d->tri[i].x3){xmin=m3d->tri[i].x3;}
  if(xmax < m3d->tri[i].x1){xmax=m3d->tri[i].x1;}
  if(xmax < m3d->tri[i].x2){xmax=m3d->tri[i].x2;}
  if(xmax < m3d->tri[i].x3){xmax=m3d->tri[i].x3;}

  if(ymin > m3d->tri[i].y1){ymin=m3d->tri[i].y1;}
  if(ymin > m3d->tri[i].y2){ymin=m3d->tri[i].y2;}
  if(ymin > m3d->tri[i].y3){ymin=m3d->tri[i].y3;}
  if(ymax < m3d->tri[i].y1){ymax=m3d->tri[i].y1;}
  if(ymax < m3d->tri[i].y2){ymax=m3d->tri[i].y2;}
  if(ymax < m3d->tri[i].y3){ymax=m3d->tri[i].y3;}

  if(zmin > m3d->tri[i].z1){zmin=m3d->tri[i].z1;}
  if(zmin > m3d->tri[i].z2){zmin=m3d->tri[i].z2;}
  if(zmin > m3d->tri[i].z3){zmin=m3d->tri[i].z3;}
  if(zmax < m3d->tri[i].z1){zmax=m3d->tri[i].z1;}
  if(zmax < m3d->tri[i].z2){zmax=m3d->tri[i].z2;}
  if(zmax < m3d->tri[i].z3){zmax=m3d->tri[i].z3;}
}

if(xmax<0){xmax=-xmax;} if(xmin<0){xmin=-xmin;} if(xmax<xmin){xmax=xmin;}
if(ymax<0){ymax=-ymax;} if(ymin<0){ymin=-ymin;} if(ymax<ymin){ymax=ymin;}
if(zmax<0){zmax=-zmax;} if(zmin<0){zmin=-zmin;} if(zmax<zmin){zmax=zmin;}

m3d->radius=sqrt(xmax*xmax+ymax*ymax+zmax*zmax);
return m3d->radius;
}


int addmod3d(){
  unsigned int i,m,n=8;
  mod3d *m3d;

i=(RGLOB_nrmd3d[0])++;
m=RGLOB_nrmd3d[1];

if(i>=m){
  m+=n;
  RGLOB_nrmd3d[1]=m;
  if(m==n){
    if(!(RGLOB_mod3d=(mod3d *)malloc(m*sizeof(mod3d)))){printf("addmod3d(): Out of memory\r\n"); exit(1);}
  }else{
    if(!(RGLOB_mod3d=(mod3d *)realloc(RGLOB_mod3d,m*sizeof(mod3d)))){printf("addmod3d(): Out of memory\r\n"); exit(1);}
  }
}

m3d=&(RGLOB_mod3d[i]);

m3d->nt[0]=0; m3d->nt[1]=n;
if(!(m3d->tri=(triangl3d *)malloc(n*sizeof(triangl3d)))){printf("addmod3d(): Out of memory\r\n"); exit(1);}

return i;
}


void freemod3d(){
  unsigned int i;
  mod3d *m3d;

if(RGLOB_nrmd3d[0] > 0){
  for(i=0;i<RGLOB_nrmd3d[0];i++){
    m3d=RGLOB_mod3d+i;
    free(m3d->tri);
  }
  RGLOB_nrmd3d[0]=0; RGLOB_nrmd3d[1]=0;
  free(RGLOB_mod3d);
}
}


void closegraph3d(){
  if(RGLOB_init3d==1){free(RGLOB_zbuf); RGLOB_init3d=0;}
  freemod3d();
  freeobj3d();
}


/*order vertices, so triangle can be culled correctly; vis='v'-visible from (x,y,z); 'i'-not visible*/
void ordercl(triangl3d *tri,char vis,float x,float y,float z){
  float dx,dy,dz,a,b,c,d,prodscal,tmp;

findplan(tri,&a,&b,&c,&d);
dx=tri->x1 - x;
dy=tri->y1 - y;
dz=tri->z1 - z;
prodscal=a*dx+b*dy+c*dz;
switch(vis){
  case 'v': tri->cull=(tri->cull)|1;
    if(prodscal<0){
      tmp=tri->x1; tri->x1=tri->x2; tri->x2=tmp;
      tmp=tri->y1; tri->y1=tri->y2; tri->y2=tmp;
      tmp=tri->z1; tri->z1=tri->z2; tri->z2=tmp;
    } break;
   case 'i': tri->cull=(tri->cull)|1;
    if(prodscal>=0){
      tmp=tri->x1; tri->x1=tri->x2; tri->x2=tmp;
      tmp=tri->y1; tri->y1=tri->y2; tri->y2=tmp;
      tmp=tri->z1; tri->z1=tri->z2; tri->z2=tmp;
    } break;
  default: break;
}

}


void addtriangle3d(triangl3d *tri){
  int nmod,i,m,n=8;
  mod3d *m3d;

  nmod=RGLOB_nrmd3d[0]-1;

  if(nmod==-1){printf("addtriangle3d(): Call addmod3d() first\r\n"); exit(1);}

  m3d=&(RGLOB_mod3d[nmod]);
  i=(m3d->nt[0])++; m=m3d->nt[1];
  if(i>=m){
    m+=n;
    m3d->nt[1]=m;
    if(!(m3d->tri=(triangl3d *)realloc(m3d->tri,m*sizeof(triangl3d)))){printf("addtriangle3d(): Out of memory\r\n"); exit(1);}
  }
  m3d->tri[i]=(*tri);
}


void mod3addtri(float x1,float y1,float z1,float x2,float y2,float z2,float x3,float y3,float z3){
  triangl3d tri;

tri.x1=x1; tri.y1=y1; tri.z1=z1;
tri.x2=x2; tri.y2=y2; tri.z2=z2;
tri.x3=x3; tri.y3=y3; tri.z3=z3;
tri.red=192.0f; tri.green=128.0f; tri.blue=64.0f; /*default colours*/
tri.cull=0;

addtriangle3d(&tri);
}


void mod3setcolor(unsigned int tr1,unsigned int tr2,float red,float green,float blue){
  int nmod;
  unsigned int j,ntri;
  mod3d *m3d;

nmod=RGLOB_nrmd3d[0]-1;
if(nmod==-1){printf("mod3setcolor(): Call addmod3d() first\r\n"); exit(1);}
m3d=&(RGLOB_mod3d[nmod]);
ntri=m3d->nt[0];

if((tr1>=ntri)||(tr2>=ntri)){
  printf("mod3setcolor(): Model has only %d triangles\r\n",ntri); exit(1);
}

for(j=tr1;j<=tr2;j++){
  m3d->tri[j].red=red; m3d->tri[j].green=green; m3d->tri[j].blue=blue;
}
}


void mod3setfullbr(unsigned int tr1,unsigned int tr2){
  int nmod;
  unsigned int j,ntri;
  mod3d *m3d;

nmod=RGLOB_nrmd3d[0]-1;
if(nmod==-1){printf("mod3setfullbr(): Call addmod3d() first\r\n"); exit(1);}
m3d=&(RGLOB_mod3d[nmod]);
ntri=m3d->nt[0];

if((tr1>=ntri)||(tr2>=ntri)){
  printf("mod3setfullbr(): Model has only %d triangles\r\n",ntri); exit(1);
}

for(j=tr1;j<=tr2;j++){(m3d->tri[j].cull)|=2;}
}


void mod3setvis(unsigned int tr1,unsigned int tr2,char vis,float xv,float yv,float zv){
  int nmod;
  unsigned int j,ntri;
  mod3d *m3d;

nmod=RGLOB_nrmd3d[0]-1;
if(nmod==-1){printf("mod3setvis(): Call addmod3d() first\r\n"); exit(1);}
m3d=&(RGLOB_mod3d[nmod]);
ntri=m3d->nt[0];

if((tr1>=ntri)||(tr2>=ntri)){
  printf("mod3setvis(): Model has only %d triangles\r\n",ntri); exit(1);
}

for(j=tr1;j<=tr2;j++){
  ordercl(&(m3d->tri[j]),vis,xv,yv,zv);
}
}


/*reads 3d model from file "fname"*/
int readmod3d(const char *fname){
  float *x,xv,yv,zv;
  char vis;
  int i,j,pt[3],w,nvert,ntri,n,nmod,tr1,tr2;
  mod3d *m3d;
  triangl3d tri;

readtok(fname,"vert:tri:f:cpoints:colors:fullbright:v:i");

w=tkgetword();
if(w==1){
  nvert=tkgetint();
}else{
  printf("'%s': 'vert' expected, found '",fname); printtok(); printf("'\r\n"); exit(1);
}

w=tkgetword();
if(w==2){
  ntri=tkgetint();
}else{
  printf("'%s': 'tri' expected, found '",fname); printtok(); printf("'\r\n"); exit(1);
}

if(!(x=(float *)malloc(3*nvert*sizeof(float)))){printf("readmod3d(): Out of memory\r\n"); exit(1);}

nmod=addmod3d();
m3d=&(RGLOB_mod3d[nmod]);

for(i=0;i<nvert;i++){for(j=0;j<3;j++){x[i*3+j]=tkgetfloat();}}
for(i=0;i<ntri;i++){
  tkgetword();
  for(j=0;j<3;j++){
    pt[j]=tkgetint() - 1;
    if(pt[j]>=nvert){printf("'%s': Vertex %d not exist\r\n",fname,pt[j]); exit(1);}
  }
  tri.x1=x[3*pt[0]]; tri.y1=x[3*pt[0]+1]; tri.z1=x[3*pt[0]+2];
  tri.x2=x[3*pt[1]]; tri.y2=x[3*pt[1]+1]; tri.z2=x[3*pt[1]+2];
  tri.x3=x[3*pt[2]]; tri.y3=x[3*pt[2]+1]; tri.z3=x[3*pt[2]+2];
  tri.red=192.0f; tri.green=128.0f; tri.blue=64.0f; /*default colours*/
  tri.cull=0;
  addtriangle3d(&tri);
}

while((w=tkgetword())){
  switch(w){
    case 4: /*cpoints*/
      n=tkgetint();
      for(i=0;i<n;i++){
        tr1=tkgetint() - 1; tr2=tkgetint(); vis=tkgetword();
        if((tr1>=ntri)||(tr2>ntri)){
          printf("'%s', cpoints: Model has only %d triangles\r\n",fname,ntri); exit(1);
        }
        if(vis==7){vis='v';}else{if(vis==8){vis='i';}}
        xv=tkgetfloat(); yv=tkgetfloat(); zv=tkgetfloat();
        for(j=tr1;j<tr2;j++){
          ordercl(&(m3d->tri[j]),vis,xv,yv,zv);
        }
      }
      break;

    case 5: /*colors*/
      n=tkgetint();
      for(i=0;i<n;i++){
        tr1=tkgetint() - 1; tr2=tkgetint();
        if((tr1>=ntri)||(tr2>ntri)){
          printf("'%s', colors: Model has only %d triangles\r\n",fname,ntri); exit(1);
        }
        xv=tkgetfloat(); yv=tkgetfloat(); zv=tkgetfloat();
        for(j=tr1;j<tr2;j++){
          m3d->tri[j].red=xv; m3d->tri[j].green=yv; m3d->tri[j].blue=zv;
        }
      }
      break;

    case 6: /*fullbright*/
      n=tkgetint();
      for(i=0;i<n;i++){
        tr1=tkgetint() - 1; tr2=tkgetint();
        if((tr1>=ntri)||(tr2>ntri)){
          printf("'%s', fullbright: Model has only %d triangles\r\n",fname,ntri); exit(1);
        }
        for(j=tr1;j<tr2;j++){
          (m3d->tri[j].cull)|=2;
        }
      }
      break;

    default: break;
  }
}

eval_mod(nmod);

free(x); freetok();
return nmod;
}


/*reads 3d model from files "fgeo", "fcol" and "fcld"*/
int rdmod3d3f(const char *fgeo,const char *fcol,const char *fcld){
  float *x,xv,yv,zv;
  char vis;
  int i,j,pt[3],w,nvert,ntri,n,nmod,tr1,tr2;
  mod3d *m3d;
  triangl3d tri;

readtok(fgeo,"f");
nvert=tkgetint(); ntri=tkgetint();
if(!(x=(float *)malloc(3*nvert*sizeof(float)))){printf("readmod3d(): Out of memory\r\n"); exit(1);}
nmod=addmod3d();
m3d=&(RGLOB_mod3d[nmod]);
for(i=0;i<nvert;i++){for(j=0;j<3;j++){x[i*3+j]=tkgetfloat();}}
for(i=0;i<ntri;i++){
  tkgetword();
  for(j=0;j<3;j++){
    pt[j]=tkgetint() - 1;
    if(pt[j]>=nvert){printf("'%s': Vertex %d not exist\r\n",fgeo,pt[j]); exit(1);}
  }
  tri.x1=x[3*pt[0]]; tri.y1=x[3*pt[0]+1]; tri.z1=x[3*pt[0]+2];
  tri.x2=x[3*pt[1]]; tri.y2=x[3*pt[1]+1]; tri.z2=x[3*pt[1]+2];
  tri.x3=x[3*pt[2]]; tri.y3=x[3*pt[2]+1]; tri.z3=x[3*pt[2]+2];
  tri.red=192.0f; tri.green=128.0f; tri.blue=64.0f; /*default colours*/
  tri.cull=0;
  addtriangle3d(&tri);
}
free(x); freetok();

if(chkname(fcld)){
readtok(fcld,"v:i");
n=tkgetint();
for(i=0;i<n;i++){
  tr1=tkgetint() - 1; tr2=tkgetint(); vis=tkgetword();
  if((tr1>=ntri)||(tr2>ntri)){
    printf("'%s', cpoints: Model has only %d triangles\r\n",fcld,ntri); exit(1);
  }
  if(vis==1){vis='v';}else{if(vis==2){vis='i';}}
  xv=tkgetfloat(); yv=tkgetfloat(); zv=tkgetfloat();
  for(j=tr1;j<tr2;j++){
    ordercl(&(m3d->tri[j]),vis,xv,yv,zv);
  }
}
freetok();
}

if(chkname(fcol)){
readtok(fcol,"fullbright");
n=tkgetint();
for(i=0;i<n;i++){
  tr1=tkgetint() - 1; tr2=tkgetint();
  if((tr1>=ntri)||(tr2>ntri)){
    printf("'%s', colors: Model has only %d triangles\r\n",fcol,ntri); exit(1);
  }
  xv=tkgetfloat(); yv=tkgetfloat(); zv=tkgetfloat();
  for(j=tr1;j<tr2;j++){
    m3d->tri[j].red=xv; m3d->tri[j].green=yv; m3d->tri[j].blue=zv;
  }
}
w=tkgetword();
if(w==1){ /*fullbright*/
  n=tkgetint();
  for(i=0;i<n;i++){
    tr1=tkgetint() - 1; tr2=tkgetint();
    if((tr1>=ntri)||(tr2>ntri)){
      printf("'%s', fullbright: Model has only %d triangles\r\n",fcol,ntri); exit(1);
    }
    for(j=tr1;j<tr2;j++){
      (m3d->tri[j].cull)|=2;
    }
  }
}
freetok();
}

eval_mod(nmod);

return nmod;
}


void light2cam(){
  float dx,dy,dz;

dx=RGLOB_light[0].dx; dy=RGLOB_light[0].dy; dz=RGLOB_light[0].dz;
RGLOB_light[1].dx=RGLOB_cam.rot[0]*dx+RGLOB_cam.rot[3]*dy+RGLOB_cam.rot[6]*dz;
RGLOB_light[1].dy=RGLOB_cam.rot[1]*dx+RGLOB_cam.rot[4]*dy+RGLOB_cam.rot[7]*dz;
RGLOB_light[1].dz=RGLOB_cam.rot[2]*dx+RGLOB_cam.rot[5]*dy+RGLOB_cam.rot[8]*dz;
}


void setobj3dmod(unsigned int i,unsigned int nmod){
  obj3d *o3d;
  mod3d *m3d;

if(i>=RGLOB_nrobj3d[0]){printf("setobj3dmod(): Object %d not exist\r\n",i); exit(1);}
if(nmod>=RGLOB_nrmd3d[0]){printf("setobj3dmod(): model %d not exist\r\n",nmod); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
  o3d->nmod=nmod;
m3d=&(RGLOB_mod3d[nmod]);
o3d->radius = m3d->radius;
}


void setobj3dpos(int i,float *pos){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("setobj3dpos(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
for(i=0;i<3;i++){o3d->pos[i]=pos[i];}
}


void setcampos(float *pos){setobj3dpos(-1,pos); light2cam();}


void setobj3drot(int i,float *rot){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("setobj3drot(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
for(i=0;i<9;i++){o3d->rot[i]=rot[i];}
}


void setcamrot(float *rot){setobj3drot(-1,rot); light2cam();}


void setobj3dscale(unsigned int i,float scale){
  float x,y,z;
  obj3d *o3d;

if(i>=RGLOB_nrobj3d[0]){printf("setobj3dscale(): Object %d not exist\r\n",i); exit(1);}
o3d=&(RGLOB_obj3d[i]);

x=o3d->rot[0]; y=o3d->rot[3]; z=o3d->rot[6];
x=sqrt(x*x+y*y+z*z); /*length of unit vector=current scale*/
y=scale/x;
for(i=0;i<9;i++){(o3d->rot[i])*=y;}
}


int addobj3d(unsigned int nmod){
  unsigned int i,m,n=8;
  obj3d *o3d;
  mod3d *m3d;

if(nmod>=RGLOB_nrmd3d[0]){printf("addobj3d(): model %d not exist\r\n",nmod); exit(1);}

i=(RGLOB_nrobj3d[0])++;
m=RGLOB_nrobj3d[1];

if(i>=m){
  m+=n;
  RGLOB_nrobj3d[1]=m;
  if(m==n){
    if(!(RGLOB_obj3d=(obj3d *)malloc(m*sizeof(obj3d)))){printf("addobj3d(): Out of memory\r\n"); exit(1);}
  }else{
    if(!(RGLOB_obj3d=(obj3d *)realloc(RGLOB_obj3d,m*sizeof(obj3d)))){printf("addobj3d(): Out of memory\r\n"); exit(1);}
  }
}

o3d=&(RGLOB_obj3d[i]);

o3d->nmod=nmod;
m3d=&(RGLOB_mod3d[nmod]);
o3d->radius = m3d->radius;

for(n=0;n<9;n++){o3d->rot[n]=0.0f;}
for(n=0;n<3;n++){o3d->pos[n]=0.0f; o3d->rot[n*3+n]=1.0f;}

return i;
}


void freeobj3d(){
  if(RGLOB_nrobj3d[0] > 0){
    free(RGLOB_obj3d); RGLOB_nrobj3d[0]=0; RGLOB_nrobj3d[1]=0;
  }
}


int drawobj3d(unsigned int nob){
  int i,n,ntri;
  float *pos,*rot,x,y,z,r;
  obj3d o3d,*o3;
  mod3d *m3d;
  triangl3d tri,*tr;

if(nob>=RGLOB_nrobj3d[0]){printf("drawobj3d(): object %d not exist\r\n",nob); exit(1);}

o3=&(RGLOB_obj3d[nob]);
ntri=0;

z=1.733f*RGLOB_zmax; r=o3->radius;
pos=RGLOB_cam.pos; rot=RGLOB_cam.rot;

if((o3->pos[0]-pos[0]-r)>z){return 0;}
if((o3->pos[0]-pos[0]+r)<(-z)){return 0;}
if((o3->pos[1]-pos[1]-r)>z){return 0;}
if((o3->pos[1]-pos[1]+r)<(-z)){return 0;}
if((o3->pos[2]-pos[2]-r)>z){return 0;}
if((o3->pos[2]-pos[2]+r)<(-z)){return 0;}

x=o3->pos[0]-pos[0]; y=o3->pos[1]-pos[1]; z=o3->pos[2]-pos[2];
o3d.pos[0]=rot[0]*x+rot[3]*y+rot[6]*z;
o3d.pos[1]=rot[1]*x+rot[4]*y+rot[7]*z;
o3d.pos[2]=rot[2]*x+rot[5]*y+rot[8]*z;

o3d.rot[0]=rot[0]*o3->rot[0]+rot[3]*o3->rot[3]+rot[6]*o3->rot[6];
o3d.rot[3]=rot[1]*o3->rot[0]+rot[4]*o3->rot[3]+rot[7]*o3->rot[6];
o3d.rot[6]=rot[2]*o3->rot[0]+rot[5]*o3->rot[3]+rot[8]*o3->rot[6];

o3d.rot[1]=rot[0]*o3->rot[1]+rot[3]*o3->rot[4]+rot[6]*o3->rot[7];
o3d.rot[4]=rot[1]*o3->rot[1]+rot[4]*o3->rot[4]+rot[7]*o3->rot[7];
o3d.rot[7]=rot[2]*o3->rot[1]+rot[5]*o3->rot[4]+rot[8]*o3->rot[7];

o3d.rot[2]=rot[0]*o3->rot[2]+rot[3]*o3->rot[5]+rot[6]*o3->rot[8];
o3d.rot[5]=rot[1]*o3->rot[2]+rot[4]*o3->rot[5]+rot[7]*o3->rot[8];
o3d.rot[8]=rot[2]*o3->rot[2]+rot[5]*o3->rot[5]+rot[8]*o3->rot[8];

z=o3d.pos[2]+r;
if(((o3d.pos[2]-r)<RGLOB_zmax)&&(z>0)){
  if((o3d.pos[1]-r)>z*RGLOB_tgh){return 0;}
  if((o3d.pos[1]+r)<z*(-RGLOB_tgh)){return 0;}
  if((o3d.pos[0]-r)>z*RGLOB_tgv){return 0;}
  if((o3d.pos[0]+r)<z*(-RGLOB_tgv)){return 0;}
  m3d=&(RGLOB_mod3d[o3->nmod]);
  n=m3d->nt[0];
  pos=o3d.pos; rot=o3d.rot;
  for(i=0;i<n;i++){
    tr=&(m3d->tri[i]);
    tri=(*tr);
    tri.x1=pos[0]+rot[0]*tr->x1+rot[1]*tr->y1+rot[2]*tr->z1;
    tri.y1=pos[1]+rot[3]*tr->x1+rot[4]*tr->y1+rot[5]*tr->z1;
    tri.z1=pos[2]+rot[6]*tr->x1+rot[7]*tr->y1+rot[8]*tr->z1;
    tri.x2=pos[0]+rot[0]*tr->x2+rot[1]*tr->y2+rot[2]*tr->z2;
    tri.y2=pos[1]+rot[3]*tr->x2+rot[4]*tr->y2+rot[5]*tr->z2;
    tri.z2=pos[2]+rot[6]*tr->x2+rot[7]*tr->y2+rot[8]*tr->z2;
    tri.x3=pos[0]+rot[0]*tr->x3+rot[1]*tr->y3+rot[2]*tr->z3;
    tri.y3=pos[1]+rot[3]*tr->x3+rot[4]*tr->y3+rot[5]*tr->z3;
    tri.z3=pos[2]+rot[6]*tr->x3+rot[7]*tr->y3+rot[8]*tr->z3;

    ntri+=dtriangle3d(&tri);
  }
}

return ntri;
}


int drawobj3all(){
  unsigned int i,nob,ntri;

nob=RGLOB_nrobj3d[0];
ntri=0;
for(i=0;i<nob;i++){ntri+=drawobj3d(i);}
return ntri;
}


void transobj3d(int i,float x,float y,float z){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("transobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
o3d->pos[0]+=x; o3d->pos[1]+=y; o3d->pos[2]+=z;
}


void transcam(float x,float y,float z){transobj3d(-1,x,y,z); light2cam();}


/*translate x,y,z in local system of object*/
void trlsobj3d(int i,float x,float y,float z){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("transobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
o3d->pos[0]+=(x*o3d->rot[0] + y*o3d->rot[1] + z*o3d->rot[2]);
o3d->pos[1]+=(x*o3d->rot[3] + y*o3d->rot[4] + z*o3d->rot[5]);
o3d->pos[2]+=(x*o3d->rot[6] + y*o3d->rot[7] + z*o3d->rot[8]);
}


void trlscam(float x,float y,float z){trlsobj3d(-1,x,y,z); light2cam();}


void rotZobj3d(int i,float tt){
  float s,c,r;
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("rotZobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
s=sin(tt); c=cos(tt);
r=o3d->rot[0];
o3d->rot[0]=c*r-s*o3d->rot[3];
o3d->rot[3]=s*r+c*o3d->rot[3];
r=o3d->rot[1];
o3d->rot[1]=c*r-s*o3d->rot[4];
o3d->rot[4]=s*r+c*o3d->rot[4];
r=o3d->rot[2];
o3d->rot[2]=c*r-s*o3d->rot[5];
o3d->rot[5]=s*r+c*o3d->rot[5];
}


void rotZcam(float tt){rotZobj3d(-1,tt); light2cam();}


void rotYobj3d(int i,float tt){
  float s,c,r;
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("rotYobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
s=sin(tt); c=cos(tt);
r=o3d->rot[0];
o3d->rot[0]=c*r+s*o3d->rot[6];
o3d->rot[6]=c*o3d->rot[6]-s*r;
r=o3d->rot[1];
o3d->rot[1]=c*r+s*o3d->rot[7];
o3d->rot[7]=c*o3d->rot[7]-s*r;
r=o3d->rot[2];
o3d->rot[2]=c*r+s*o3d->rot[8];
o3d->rot[8]=c*o3d->rot[8]-s*r;
}


void rotYcam(float tt){rotYobj3d(-1,tt); light2cam();}


void rotXobj3d(int i,float tt){
  float s,c,r;
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("rotXobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
s=sin(tt); c=cos(tt);
r=o3d->rot[3];
o3d->rot[3]=c*r-s*o3d->rot[6];
o3d->rot[6]=s*r+c*o3d->rot[6];
r=o3d->rot[4];
o3d->rot[4]=c*r-s*o3d->rot[7];
o3d->rot[7]=s*r+c*o3d->rot[7];
r=o3d->rot[5];
o3d->rot[5]=c*r-s*o3d->rot[8];
o3d->rot[8]=s*r+c*o3d->rot[8];
}


void rotXcam(float tt){rotXobj3d(-1,tt); light2cam();}


void rotABobj3d(int i,float dx,float dy,float dz,float tt){
  float sa,ca,sb,cb,st,ct,r,l1,l2;
  char ax;
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("rotABobj3d(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}

st=dx*dx; ct=dy*dy; r=dz*dz; l2=sqrt(st+ct+r);

if(st>r){
  ax='z'; l1=sqrt(st+ct); sa=dy/l1; ca=dx/l1; sb=l1/l2; cb=dz/l2;
}else{
  ax='x'; l1=sqrt(ct+r); sa=-dy/l1; ca=dz/l1; sb=dx/l2; cb=l1/l2;
}

st=sin(tt); ct=cos(tt);

if(ax=='z'){
  r=o3d->rot[0];
  o3d->rot[0]=ca*r+sa*o3d->rot[3];
  o3d->rot[3]=ca*o3d->rot[3]-sa*r;
  r=o3d->rot[1];
  o3d->rot[1]=ca*r+sa*o3d->rot[4];
  o3d->rot[4]=ca*o3d->rot[4]-sa*r;
  r=o3d->rot[2];
  o3d->rot[2]=ca*r+sa*o3d->rot[5];
  o3d->rot[5]=ca*o3d->rot[5]-sa*r;
}else{
  r=o3d->rot[3];
  o3d->rot[3]=ca*r+sa*o3d->rot[6];
  o3d->rot[6]=ca*o3d->rot[6]-sa*r;
  r=o3d->rot[4];
  o3d->rot[4]=ca*r+sa*o3d->rot[7];
  o3d->rot[7]=ca*o3d->rot[7]-sa*r;
  r=o3d->rot[5];
  o3d->rot[5]=ca*r+sa*o3d->rot[8];
  o3d->rot[8]=ca*o3d->rot[8]-sa*r;
}

r=o3d->rot[0];
o3d->rot[0]=cb*r-sb*o3d->rot[6];
o3d->rot[6]=cb*o3d->rot[6]+sb*r;
r=o3d->rot[1];
o3d->rot[1]=cb*r-sb*o3d->rot[7];
o3d->rot[7]=cb*o3d->rot[7]+sb*r;
r=o3d->rot[2];
o3d->rot[2]=cb*r-sb*o3d->rot[8];
o3d->rot[8]=cb*o3d->rot[8]+sb*r;

r=o3d->rot[0];
o3d->rot[0]=ct*r-st*o3d->rot[3];
o3d->rot[3]=st*r+ct*o3d->rot[3];
r=o3d->rot[1];
o3d->rot[1]=ct*r-st*o3d->rot[4];
o3d->rot[4]=st*r+ct*o3d->rot[4];
r=o3d->rot[2];
o3d->rot[2]=ct*r-st*o3d->rot[5];
o3d->rot[5]=st*r+ct*o3d->rot[5];

r=o3d->rot[0];
o3d->rot[0]=cb*r+sb*o3d->rot[6];
o3d->rot[6]=cb*o3d->rot[6]-sb*r;
r=o3d->rot[1];
o3d->rot[1]=cb*r+sb*o3d->rot[7];
o3d->rot[7]=cb*o3d->rot[7]-sb*r;
r=o3d->rot[2];
o3d->rot[2]=cb*r+sb*o3d->rot[8];
o3d->rot[8]=cb*o3d->rot[8]-sb*r;

if(ax=='z'){
  r=o3d->rot[0];
  o3d->rot[0]=ca*r-sa*o3d->rot[3];
  o3d->rot[3]=ca*o3d->rot[3]+sa*r;
  r=o3d->rot[1];
  o3d->rot[1]=ca*r-sa*o3d->rot[4];
  o3d->rot[4]=ca*o3d->rot[4]+sa*r;
  r=o3d->rot[2];
  o3d->rot[2]=ca*r-sa*o3d->rot[5];
  o3d->rot[5]=ca*o3d->rot[5]+sa*r;
}else{
  r=o3d->rot[3];
  o3d->rot[3]=ca*r-sa*o3d->rot[6];
  o3d->rot[6]=ca*o3d->rot[6]+sa*r;
  r=o3d->rot[4];
  o3d->rot[4]=ca*r-sa*o3d->rot[7];
  o3d->rot[7]=ca*o3d->rot[7]+sa*r;
  r=o3d->rot[5];
  o3d->rot[5]=ca*r-sa*o3d->rot[8];
  o3d->rot[8]=ca*o3d->rot[8]+sa*r;
}

}


void rotABcam(float dx,float dy,float dz,float tt){
  rotABobj3d(-1,dx,dy,dz,tt);
  light2cam();
}


void drawfog3d(float zfog,float zmax){
  unsigned int i,l,height;
  float r,t,izmax0,izmax,izfog;
  Uint8 red,green,blue,bred,bgreen,bblue;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpinc;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpinc;
  #endif

RGLOB_zmax=zmax;
if((zfog<=RGLOB_zmin)||(zfog>=RGLOB_zmax)){
  printf("drawfog(): zfog=%1.3f not in (zmin=%1.3f,zmax=%1.3f)\r\n",zfog,RGLOB_zmin,RGLOB_zmax);
  exit(1);
}

#if DOUBLEPIX==1
  height=RGLOB_height>>1;
#else
  height=RGLOB_height;
#endif

SDL_GetRGB(RGLOB_color[1],RGLOB_pixformdet,NULL,&bred,&bgreen,&bblue);

izmax0=1.0f/RGLOB_zmax;
izmax=1.001f*izmax0;
izfog=1.0f/zfog;
r=1.0f/(izmax-izfog);

l=0;
for(i=0;i<height;i++){

  #if DOUBLEPIX==0
    ptrmax=pl[i]+RGLOB_width;
    for(ptr=pl[i];ptr<ptrmax;ptr++)
  #else
    ptrmax=pl[i<<1]+RGLOB_width;
    for(ptr=pl[i<<1];ptr<ptrmax;ptr+=2)
  #endif
    {
      if(RGLOB_zbuf[l]<=izmax){
        (*ptr)=RGLOB_color[1];
      }else{
        if(RGLOB_zbuf[l]<=izfog){

          SDL_GetRGB(*ptr,RGLOB_pixformdet,NULL,&red,&green,&blue);

          t=(RGLOB_zbuf[l]-izfog)*r;
          red+=(int)(t*(bred-red));
          green+=(int)(t*(bgreen-green));
          blue+=(int)(t*(bblue-blue));

          (*ptr)=SDL_MapRGB(RGLOB_pixformdet,NULL,red,green,blue);
        }
      }
    RGLOB_zbuf[l]=izmax0;
    l++;

    #if DOUBLEPIX==1
      *(ptr+1)=*ptr;
    #endif
  }
}

#if DOUBLEPIX==1
  for(i=0;i<height;i++){
    ptr=pl[i<<1]; ptrmax=pl[(i<<1)+1];
    for(l=0;l<RGLOB_width;l++){
      *ptrmax = *ptr;
      ptr++; ptrmax++;
    }
  }
#endif
}


void drawbkg3d(float zmax){
  unsigned int i,l,height;
  float izmax0,izmax;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpinc;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpinc;
  #endif

RGLOB_zmax=zmax;

#if DOUBLEPIX==1
  height=RGLOB_height>>1;
#else
  height=RGLOB_height;
#endif

izmax0=1.0f/RGLOB_zmax;
izmax=1.001f*izmax0;

l=0;
for(i=0;i<height;i++){

  #if DOUBLEPIX==0
    ptrmax=pl[i]+RGLOB_width;
    for(ptr=pl[i];ptr<ptrmax;ptr++)
  #else
    ptrmax=pl[i<<1]+RGLOB_width;
    for(ptr=pl[i<<1];ptr<ptrmax;ptr+=2)
  #endif
    {
      if(RGLOB_zbuf[l]<=izmax){
        (*ptr)=RGLOB_color[1];
      }
    RGLOB_zbuf[l]=izmax0;
    l++;

    #if DOUBLEPIX==1
      *(ptr+1)=*ptr;
    #endif
  }
}

#if DOUBLEPIX==1
  for(i=0;i<height;i++){
    ptr=pl[i<<1]; ptrmax=pl[(i<<1)+1];
    for(l=0;l<RGLOB_width;l++){
      *ptrmax = *ptr;
      ptr++; ptrmax++;
    }
  }
#endif
}


float *getobj3dpos(int i){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("setobj3dpos(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
return o3d->pos;
}


float *getcampos(){return getobj3dpos(-1);}


float *getobj3drot(int i){
  obj3d *o3d;
if(i<0){
  o3d=&RGLOB_cam;
}else{
  if(i>=(int)RGLOB_nrobj3d[0]){printf("setobj3drot(): Object %d not exist\r\n",i); exit(1);}
  o3d=&(RGLOB_obj3d[i]);
}
return o3d->rot;
}


float *getcamrot(){return getobj3drot(-1);}


/*multiply colours of all triangles with fred, fgreen and fblue*/
void clfactors(float fred,float fgreen,float fblue){
  unsigned int i;
  int j;
  mod3d *mod;
for(i=0;i<RGLOB_nrmd3d[0];i++){
  mod=&(RGLOB_mod3d[i]); /*RGLOB_mod3d+i*/
  for(j=0;j<(mod->nt[0]);j++){
    (mod->tri[j].red)*=fred;
    (mod->tri[j].green)*=fgreen;
    (mod->tri[j].blue)*=fblue;
  }
}
}
